<?php

/**
 * Class to register meta box.
 */
class ThemeRain_Meta_Box {

	/**
	 * Constructor.
	 */
	public function __construct( $meta_box ) {

		$this->meta_box = $meta_box;

		// Add meta box
		add_action( 'add_meta_boxes', array( $this, 'add' ) );

		// Save meta box
		add_action( 'save_post', array( $this, 'save' ) );
	}

	// Add meta box.
	public function add() {

		add_meta_box(
			'themerain_meta_' . $this->meta_box['id'],
			$this->meta_box['title'],
			array( $this, 'render' ),
			$this->meta_box['screen'],
			isset( $this->meta_box['context'] ) ? $this->meta_box['context'] : 'normal',
			isset( $this->meta_box['context'] ) ? 'default' : 'high'
		);
	}

	// Render meta box.
	public function render( $post ) {

		wp_nonce_field( $this->meta_box['id'], $this->meta_box['id'] . '_nonce' );

		if ( isset( $this->meta_box['template'] ) ) {
			echo '<div class="themerain-condition-template themerain-hidden" data-id="themerain_meta_' . $this->meta_box['id'] . '" data-template="' . $this->meta_box['template'] . '"></div>';
		}

		if ( isset( $this->meta_box['page_template'] ) ) {
			echo '<div class="themerain-condition-template themerain-hidden" data-id="themerain_meta_' . $this->meta_box['id'] . '" data-template="' . $this->meta_box['page_template'] . '"></div>';
		}

		foreach ( $this->meta_box['fields'] as $field ) {
			$this->field( $post, $field );
		}
	}

	// Fields.
	public function field( $post, $field ) {

		$meta = get_post_meta( $post->ID, $field['id'], true );
		$std  = isset( $field['std'] ) ? $field['std'] : '';
		$meta = ( $meta || '0' === $meta ) ? $meta : $std;
		$id   = $field['id'] ? esc_attr( $field['id'] ) : '';

		$field_classes   = array();
		$field_classes[] = 'themerain-meta-field';
		$field_classes[] = 'themerain-meta-' . $field['type'];

		$condition = '';
		if ( isset( $field['cond'] ) || isset( $field['condition'] ) ) {
			if ( isset( $field['cond'] ) ) {
				$cond = $field['cond'];
			} elseif ( isset( $field['condition'] ) ) {
				$cond = $field['condition'];
			}
			$cond_meta = get_post_meta( $post->ID, $cond[0], true );
			$condition = ' data-condition="' . implode( ',', $cond ) . '"';

			if ( '==' === $cond[1] ) {
				$field_classes[] = ( $cond[2] === $cond_meta ) ? '' : 'themerain-hidden';
			} elseif ( '!=' === $cond[1] ) {
				$field_classes[] = ( $cond[2] !== $cond_meta ) ? '' : 'themerain-hidden';
			}
		}

		$html = '<div class="' . implode( ' ', $field_classes ) . '"' . $condition . '>';

		$html .= '<div class="themerain-meta-label">';
		$html .= '<label for="' . $id . '">' . $field['label'] . '</label>';
		if ( isset( $field['desc'] ) ) {
			$html .= '<p class="description">' . $field['desc'] . '</p>';
		}
		$html .= '</div>';

		$html .= '<div class="themerain-meta-input">';

		switch ( $field['type'] ) {

			case 'text':
				$html .= '<input type="text" id="' . $id . '" name="' . $id . '" value="' . wp_kses_post( $meta ) . '">';
				break;

			case 'textarea':
				$html .= '<textarea id="' . $id . '" name="' . $id . '" rows="5">' . wp_kses_post( $meta ) . '</textarea>';
				break;

			case 'url':
				$html .= '<div class="themerain-meta-url__wrap">';
				$html .= '<i class="themerain-meta-url__icon"></i>';
				$html .= '<input type="url" id="' . $id . '" name="' . $id . '" value="' . esc_url( $meta ) . '">';
				$html .= '</div>';
				break;

			case 'checkbox':
				$html .= '<input type="checkbox" id="' . $id . '" name="' . $id . '"' . ( $meta ? ' checked' : '' ) . '>';
				break;

			case 'toggle':
				$html .= '<div class="themerain-toggle' . ( $meta ? ' is-checked' : '' ) . '">';
				$html .= '<input type="checkbox" id="' . $id . '" name="' . $id . '"' . ( $meta ? ' checked' : '' ) . '>';
				$html .= '<div class="themerain-toggle__track"></div><div class="themerain-toggle__slider"></div>';
				$html .= '</div>';
				break;

			case 'select':
				$html .= '<select id="' . $id . '" name="' . $id . '">';
				foreach( $field['choices'] as $value => $label ) {
					$html .= '<option value="' . esc_attr( $value ) . '"' . ( $meta === $value ? ' selected' : '' ) . '>' . esc_html( $label ) . '</option>';
				}
				$html .= '</select>';
				break;

			case 'color':
				$html .= '<input type="text" id="' . $id . '" name="' . $id . '" value="' . sanitize_hex_color( $meta ) . '">';
				break;

			case 'range':
				$range = isset( $field['range'] ) ? ' min="' . $field['range'][0] . '" max="' . $field['range'][1] . '" step="' . $field['range'][2] . '"' : '';

				$html .= '<input type="range" id="' . $id . '" name="' . $id . '" value="' . esc_attr( $meta ) . '"' . $range . '>';
				$html .= '<span class="themerain-meta-range__value">' . esc_attr( $meta ) . '</span>';
				break;

			case 'group':
				foreach ( $field['choices'] as $value => $label ) {
					$html .= '<label class="button' . ( $value === $meta ? ' button-primary' : '' ) . '">';
					$html .= '<input type="radio" name="' . $id . '" value="' . esc_attr( $value ) . '"' . ( $value === $meta ? ' checked' : '' ) . '>';
					$html .= $label . '</label> ';
				}
				break;

			case 'media':
				if ( isset( $field['media'] ) || isset( $field['media_type'] ) ) {
					$media     = $meta ? '<div></div>' : '';
					$data_type = 'video/mp4';
					$btn_text  = 'Upload video';
				} else {
					$media     = $meta ? '<img src="' . wp_get_attachment_image_url( esc_attr( $meta ) ) . '">' : '';
					$data_type = 'image';
					$btn_text  = 'Upload image';
				}

				$html .= '<div class="themerain-meta-media__wrap' . ( $meta ? ' has-value' : '' ) . '">';
				$html .= '<input type="hidden" name="' . $id . '" value="' . esc_attr( $meta ) . '" data-type="' . $data_type . '">';
				$html .= '<a href="#" class="themerain-meta-media__preview">' . $media . '</a>';
				$html .= '<a href="#" class="themerain-meta-media__remove" title="Remove"></a>';
				$html .= '<button class="button themerain-meta-media__upload">' . $btn_text . '</button>';
				$html .= '</div>';
				break;

			case 'pages':
				$dropdown = wp_dropdown_pages(
					array(
						'name'              => $id,
						'echo'              => 0,
						'show_option_none'  => '&mdash; Select &mdash;',
						'option_none_value' => '0',
						'selected'          => $meta
					)
				);

				$html .= $dropdown;
				break;

			case 'taxonomy':
				$tax   = isset( $field['taxonomy'] ) ? $field['taxonomy'] : 'project-category';
				$terms = get_terms( $tax, array( 'hide_empty' => false ) );

				if ( $terms ) {
					$html .= '<select id="' . $id . '" name="' . $id . '[]" multiple>';
						foreach( $terms as $term ) {
							$selected = ( is_array( $meta ) && in_array( $term->term_id, $meta ) ) ? ' selected' : '';
							$html .= '<option value="' . esc_attr( $term->term_id ) . '"' . $selected . '>' . esc_html( $term->name ) . '</option>';
						}
					$html .= '</select>';
				} else {
					$html .= 'No categories found.';
				}
				break;
		}

		$html .= '</div>';
		$html .= '</div>';

		echo $html;
	}

	// Save meta box.
	public function save( $post_id ) {

		$is_autosave    = wp_is_post_autosave( $post_id );
		$is_revision    = wp_is_post_revision( $post_id );
		$nonce_name     = $this->meta_box['id'] . '_nonce';
		$is_valid_nonce = ( isset( $_POST[$nonce_name] ) && wp_verify_nonce( $_POST[$nonce_name], $this->meta_box['id'] ) ) ? true : false;

		if ( $is_autosave || $is_revision || ! $is_valid_nonce ) {
			return;
		}

		foreach( $this->meta_box['fields'] as $field ) {
			$new = isset( $_POST[$field['id']] ) ? $_POST[$field['id']] : '';
			update_post_meta( $post_id, $field['id'], $new );
		}
	}
}
